import java.text.*;
import java.util.*;

public class Scheduler 
{
	public static double findNextPnEpsilon = 1e-7;
	public static double calcPbEpsilon = 1e-7;
	public static double lowerBoundPn = 1e-5;
	public static double upperBoundPn = 10;
	public static double energyLevelGranularity = 1e-3;

	public static DecimalFormat timeFormat = new DecimalFormat("#.###");
	public static DecimalFormat powerFormat = new DecimalFormat("#.###");
	public static DecimalFormat energyFormat = new DecimalFormat("#.###");

	private class Solution	
	{
		Config[] phi = null;		// the configuration
		boolean[] b = null;			// endpoints of closed groups
		double[] p = null;			// power
		double[] r = null;			// release times
		double[] s = null;			// start times
		double[] C = null;			// completion times
		int n;

		public Solution(int n)
		{
			this.n = n;
		}

		public double getTotalEnergy()
		{
			double E = 0.0;
			for (int i = 0; i < p.length; i++)
				E += Math.pow(p[i], (alpha - 1) / alpha);
			return E;			
		}

		public double getTotalCompletionTime()
		{
			double x = 0.0;
			for (int i = 0; i < p.length; i++)
				x += C[i];
			return x;			
		}

		public void setPhi(Config[] phi)
		{
			this.phi = new Config[phi.length];
			for (int i = 0; i < phi.length; i++)
				this.phi[i] = phi[i];
		}
	}

	private class ConfigurationInterval
	{
		final double upperPn;
		final Config[] phi;

		ConfigurationInterval(double upperPn, Config[] phi)
		{
			this.upperPn = upperPn;
			this.phi = new Config[phi.length];
			for (int i = 0; i < phi.length; i++)
				this.phi[i] = phi[i];
		}
	}

	public enum Config { EQ { public String toString() { return "="; } },
						 LT { public String toString() { return "<"; } },
						 GT { public String toString() { return ">"; } } };

	public double alpha = 3.0;

	private ArrayList<ConfigurationInterval> configurationIntervals = null;
	double[] energyLevels = null;
	double[] objective = null;
	int[] configurationIndices = null;

	void debugOutput(String s)
	{
		System.out.println(s);
	}

	public void calculate(Collection<Job> jobCollection)
	{
		ArrayList<Job> jobs = new ArrayList<Job>();
		jobs.addAll(jobCollection);
		Collections.sort(jobs);

		int n = jobs.size();

		// start with all < configuration
		Config[] phi = new Config[n-1];
		for (int i = 0; i < n-1; i++)
			phi[i] = Config.LT;

		debugOutput("Calculating optimal schedules for jobs with release dates:");
		String relDates = "";
		for (int i = 0; i < n; i++)
			relDates += jobs.get(i).releaseDate + " ";
		debugOutput(relDates);

		double pn = upperBoundPn;

		configurationIntervals = new ArrayList<ConfigurationInterval>();
		configurationIntervals.add(new ConfigurationInterval(pn, phi));

		while (pn > lowerBoundPn)
		{
			// invariant: the current configuration is optimal for 
			//            the current value of pn
			pn = findNextPn(jobs, phi, pn);

			// update violated constraints
			Solution sol = scheduleFixedConfiguration(jobs, phi, pn);
			boolean[] violated = getViolations(sol);
			for (int i = 0; i < violated.length; i++)
			{
				if (!violated[i]) continue;

				switch (phi[i])
				{	
					case LT: phi[i] = Config.EQ; break;
					case EQ: phi[i] = Config.GT; break;
					case GT: debugOutput("GT constraint violated: this should not happen"); break;
				}
			}

			configurationIntervals.add(new ConfigurationInterval(pn, phi));

			sol = scheduleFixedConfiguration(jobs, phi, pn);
			debugOutput("Next pn = " + timeFormat.format(pn) + "; " + 
						"new configuration = " + configurationString(phi) + "; " +
						"energy = " + energyFormat.format(sol.getTotalEnergy()));
			printSolution(jobs, sol);
		}

		constructEnergyLevels(jobs);
	}

	private void constructEnergyLevels(ArrayList<Job> jobs)
	{
		int N = 1 + (int) ((upperBoundPn - lowerBoundPn) / energyLevelGranularity);
		energyLevels = new double[N];
		objective = new double[N];
		configurationIndices = new int[N];
		int configurationIndex = 0;
		Config[] phi = configurationIntervals.get(0).phi;
		for (int i = energyLevels.length - 1; i >= 0; i--)
		{
			double pn = lowerBoundPn + i * energyLevelGranularity;
			while ((configurationIndex < configurationIntervals.size()) && (pn < configurationIntervals.get(configurationIndex+1).upperPn))
			{
				configurationIndex++;
				phi = configurationIntervals.get(configurationIndex).phi;
//				System.out.println("Moving to new configuration at pn = " + pn);
			}

			Solution sol = scheduleFixedConfiguration(jobs, phi, pn);
			energyLevels[i] = sol.getTotalEnergy();
			objective[i] = sol.getTotalCompletionTime();
			configurationIndices[i] = configurationIndex;
			assert (!containsViolations(sol));
//			System.out.println("pn = " + pn + " corresponds to energy level " + sol.getTotalEnergy() + " level " + configurationIndex);
		}
	}

	public String configurationString(Config[] phi)
	{
		String s = "";
		for (int i = 0; i < phi.length; i++)
			s += phi[i].toString();
		return s;
	}

	private double findNextPn(ArrayList<Job> jobs, Config[] phi, double pn)
	{
		double pnA = lowerBoundPn;
		double pnB = pn;
		while (pnB - pnA > findNextPnEpsilon)
		{
			double pnM = (pnA + pnB) / 2.0;
			Solution sol = scheduleFixedConfiguration(jobs, phi, pnM);

			if (containsViolations(sol))
				pnA = pnM;
			else
				pnB = pnM;

		}
		return pnA;
	}

	public void printSolution(ArrayList<Job> jobs, Solution sol)
	{
		for (int i = 0; i < jobs.size(); i++)
		{
			Job job = jobs.get(i);
			System.out.println("Job " + job.id + " starts at " + timeFormat.format(sol.s[i]) + ", completes at " + timeFormat.format(sol.C[i]) + 
				" (power=" + powerFormat.format(sol.p[i]) + ")");
		}

		boolean[] violated = getViolations(sol);
		int vcount = 0;
		for (int i = 0; i < violated.length; i++)
		{
			if (!violated[i]) continue;
			System.out.println("phi[" + i + "] violates one of the constraints");
			vcount++;
		}
		if (vcount == 0)
			System.out.println("No violations");
		
	}


	public void setEnergyLevel(ArrayList<Job> jobs, double energyLevel)
	{
		// find corresponding value of pn by bisection. 
		// Recall that the energylevels array is increasing
		int a = 0;
		int b = energyLevels.length - 1;
		while (b - a > 1)
		{
			int m = (a+b)/2;
			if (energyLevels[m] > energyLevel)
				b = m;
			else
				a = m;
		}

		double pn = lowerBoundPn + b * energyLevelGranularity;
		Config[] phi = configurationIntervals.get(configurationIndices[a]).phi;
		// debugOutput("Energy level set = " + energyLevel + ", acquired = " + energyLevels[a] + ", pn = " + pn);

		Solution sol = scheduleFixedConfiguration(jobs, phi, pn);
		activateSchedule(jobs, sol.p, phi);
	}

	public void graphEnergyLevels(GraphPanel graphPanel)
	{
		double[] pn = new double[energyLevels.length];
		for (int i = 0; i < energyLevels.length; i++)
			pn[i] = lowerBoundPn + i * energyLevelGranularity;

		graphPanel.Title = "Total energy usage vs. p[n]";
		graphPanel.XLabel = "Total energy usage";
		graphPanel.YLabel = "p[n]";
		graphPanel.setData(energyLevels, pn);
	}

	public void graphCompletionDates(GraphPanel graphPanel)
	{
		graphPanel.Title = "Total energy usage vs. total completion time";
		graphPanel.XLabel = "Total energy usage";
		graphPanel.YLabel = "Total completion time";
		graphPanel.setData(energyLevels, objective);
	}

	private void activateSchedule(ArrayList<Job> jobs, double[] p, Config[] phi)
	{
		for (int i = 0; i < jobs.size(); i++)
		{
			Job job = jobs.get(i);
			job.power = p[i];
			job.speed = Math.pow(p[i], 1.0 / alpha);
			job.processingTime = 1.0 / job.speed;
			job.energyUse = p[i] * job.processingTime;
			job.config = (i == jobs.size() - 1) ? "" : phi[i].toString();
		}

		double t = 0;
		for (int i = 0; i < jobs.size(); i++)
		{
			Job job = jobs.get(i);
			job.startDate = Math.max(t, job.releaseDate);
			job.completionDate = job.startDate + job.processingTime;
			t = job.completionDate;
		}
	}

	public Solution scheduleFixedConfiguration(ArrayList<Job> jobs, Config[] phi, double pn)
	{
		int n = jobs.size();
		Solution sol = new Solution(n);

		double[] p = new double[n];
		boolean[] bs = new boolean[n];
		for (int i = 0; i < n; i++)
			bs[i] = false;

		p[n-1] = pn;

		// determine maximal substrings of > jobs
		int b = n-1;
		while (b >= 0) 
		{
			// determine maximal substring of > jobs, starting at a
			int a = b;

			while ((a > 0) && (phi[a-1] == Config.GT))
				a--;

			// see if the substring is open or closed
			if ((b == n-1) || (phi[b] == Config.LT))
			{
				// "open" string
				// debugOutput("Open maximal substring runs from " + a + " to " + b);

				p[b] = pn;
				for (int i = b - 1; i >= a; i--)
					p[i] = p[b] + (b-i) * p[n-1]; 
			}
			else 
			{
				// debugOutput("Closed maximal substring runs from " + a + " to " + b);
				assert(phi[b] != Config.GT);

				bs[b] = true;

				p[b] = calcPb(p[n-1], a, b, jobs.get(b+1).releaseDate, jobs.get(a).releaseDate);
				for (int i = b - 1; i >= a; i--)
					p[i] = p[b] + (b-i) * p[n-1]; 
			}		

			if (a == 0)
				break;

			if (b == a)
				b = a - 1;
			else if (phi[a] == Config.GT)
				b = a - 1;
			else
 				b = a;
		}


		// calculate completion times
		double t = 0;
		double[] C = new double[n];
		double[] s = new double[n];
		double[] r = new double[n];
		for (int i = 0; i < jobs.size(); i++)
		{
			Job job = jobs.get(i);
			r[i] = job.releaseDate;
			t =  Math.max(t, r[i]);
			s[i] = t;
			t += Math.pow(p[i], -1.0/alpha);
			C[i] = t;
		}		

		sol.setPhi(phi);
		sol.p = p;
		sol.s = s;
		sol.C = C;
		sol.b = bs;
		sol.r = r;
		return sol;
	}

	private boolean containsViolations(Solution sol)
	{
		boolean[] violated = getViolations(sol);
		for (int i = 0; i < violated.length; i++)
			if (violated[i]) 
				return true;
		return false;
	}

	// check the conditions from Section 3.7
	private boolean[] getViolations(Solution sol)
	{
		boolean[] violations = new boolean[sol.n-1];

		// check < constraints
		for (int i = 0; i < sol.n-1; i++)
			if ((sol.phi[i] == Config.LT) && (sol.C[i] >= sol.r[i+1]))
				violations[i] = true;

		// check powers in closed groups
		for (int i = 0; i < sol.n-1; i++)
		{
			if (!sol.b[i])
				continue;

			if (sol.p[i] > sol.p[i+1] + sol.p[sol.n-1])
				violations[i] = true;
		}

		return violations;
	}


	// solves the equation
	// sum_{i=a}^b (p[b] + (b-i)p[n-1])^{-1/a} = x1 - x2
	// by means of bisection
	public double calcPb(double pn, int a, int b, double x1, double x2)
	{
		double p1 = 0;
		double p2 = 1000;
		while ( (p2 - p1) > calcPbEpsilon)
		{
			double pb = (p1+p2)/2.0;
			if (calcPb_fun(pb, pn, a, b, x1, x2) > 0)
				p1 = pb;
			else
				p2 = pb;
		}
		return (p1+p2)/2.0;
	}

	public double calcPb_fun(double pb, double pn, int a, int b, double x1, double x2)
	{
		double x = x2 - x1;
		for (int i = a; i <= b; i++)
			x += Math.pow(pb + (b-i) * pn, -1.0/alpha);
		return x;
	}
}


